<?php

namespace App\Models;

use App\Models\Traits\Blameable;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Facades\Auth;

class Complaint extends Model
{
    use SoftDeletes, Blameable;

    protected $fillable = [
        'team_id',
        'resident_id',
        'title',
        'description',
        'attachment',
        'status',
    ];

    protected $casts = [
        'attachment' => 'array',
    ];

    public function team()
    {
        return $this->belongsTo(Team::class);
    }

    public function resident()
    {
        return $this->belongsTo(Resident::class);
    }

    public function histories()
    {
        return $this->hasMany(ComplaintHistory::class);
    }

    /**
     * Mapping status aduan.
     */
    public const STATUS = [
        0 => 'MENUNGGU',
        1 => 'DIPROSES',
        2 => 'SELESAI',
        3 => 'DITOLAK',
    ];

    public function getStatusLabelAttribute(): string
    {
        return self::STATUS[$this->status] ?? '-';
    }

    protected static function booted()
    {
        static::creating(function ($complaint) {
            if (!$complaint->code) {
                $lastId = Complaint::max('id') + 1;

                // Format lebih rapi, misal CMP-2025-001
                $complaint->code = 'CMP-'
                    . now()->format('Y')
                    . '-'
                    . str_pad($lastId, 3, '0', STR_PAD_LEFT);
            }
        });

        static::created(function ($complaint) {
            $complaint->histories()->create([
                'status'     => 0,
                'note'       => 'Aduan dibuat',
                'created_by' => Auth::id(),
            ]);
        });
    }

    public function addHistory(int $status, ?string $note = null)
    {
        return $this->histories()->create([
            'status'  => $status,
            'note'    => $note,
            'created_by' => Auth::id(),
        ]);
    }
}
