<?php

namespace App\Models;

use App\Models\Traits\Blameable;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class OutgoingLetter extends Model
{
    use SoftDeletes, Blameable;

    protected $fillable = [
        'team_id',
        'letter_number',
        'letter_date',
        'resident_id',
        'recipient_name',
        'recipient_address',
        'letter_type_id',
        'letter_request_id',
        'subject',
        'content',
        'attachments',
        'status',
        'created_by',
        'updated_by',
        'deleted_by',
    ];

    protected $casts = [
        'letter_date' => 'date',
        'attachments' => 'array',
    ];

    public function team()
    {
        return $this->belongsTo(Team::class);
    }

    // Relasi
    public function resident()
    {
        return $this->belongsTo(Resident::class);
    }

    public function type()
    {
        return $this->belongsTo(LetterType::class, 'letter_type_id');
    }

    public function request()
    {
        return $this->belongsTo(LetterRequest::class, 'letter_request_id');
    }

    /**
     * Generate nomor surat baru untuk team & jenis tertentu
     */
    protected static function generateNumber(int $teamId, int $letterTypeId): string
    {
        $year = now()->year;
        $month = now()->month;

        $sequence = LetterNumberSequence::firstOrCreate(
            [
                'team_id'      => $teamId,
                'letter_type_id' => $letterTypeId,
                'year'           => $year,
                'month'           => $month,
            ],
            ['last_number' => 0]
        );

        $sequence->increment('last_number');
        $nextNumber = $sequence->last_number;

        $romanMonths = [
            1 => 'I',
            2 => 'II',
            3 => 'III',
            4 => 'IV',
            5 => 'V',
            6 => 'VI',
            7 => 'VII',
            8 => 'VIII',
            9 => 'IX',
            10 => 'X',
            11 => 'XI',
            12 => 'XII'
        ];

        $month = $romanMonths[now()->month] ?? now()->month;

        $letterType = LetterType::find($letterTypeId);

        return str_pad($nextNumber, 3, '0', STR_PAD_LEFT)
            . '/' . strtoupper($letterType->code)
            . '/' . $month
            . '/' . $year;
    }

    /**
     * Buat surat keluar dari permohonan
     */
    public static function createFromRequest(LetterRequest $request): self
    {
        return DB::transaction(function () use ($request) {
            $letterNumber = self::generateNumber(
                $request->team_id,
                $request->letter_type_id
            );

            $outgoing = self::create([
                'team_id'         => $request->team_id,
                'letter_number'     => $letterNumber,
                'letter_date'       => now(),
                'recipient_name'  => $request->resident->name,
                'recipient_address' => $request->resident->address,
                'letter_type_id'    => $request->letter_type_id,
                'letter_request_id' => $request->id,
                'subject'           => $request->purpose,
                'status'            => 0,
                'created_by'        => Auth::id(),
            ]);

            $request->update([
                'status' => 2,
            ]);

            LetterRequestHistory::create([
                'team_id'         => $request->team_id,
                'letter_request_id' => $request->id,
                'status'            => 2,
                'note'              => "Surat Anda sedang dicetak dan menunggu tanda tangan",
                'created_by'        => Auth::id(),
            ]);

            return $outgoing;
        });
    }
}
