<?php

namespace App\Models;

use App\Models\Traits\Blameable;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Facades\Auth;

class LetterRequest extends Model
{
    use SoftDeletes, Blameable;

    protected $fillable = [
        'team_id',
        'resident_id',
        'letter_type_id',
        'request_number',
        'purpose',
        'status',
        'requirement_files',
        'draft_file',
        'outgoing_letter_id',
        'created_by',
        'updated_by',
        'deleted_by',
    ];

    protected $casts = [
        'requirement_files' => 'array', // JSON → array
    ];

    /**
     * Status mapping.
     */
    const STATUS = [
        0 => 'Menunggu Verifikasi',
        1 => 'Diproses (Draft Dibuat)',
        2 => 'Menunggu Tanda Tangan',
        3 => 'Selesai (Final Upload)',
        9 => 'Ditolak',
    ];

    public function team()
    {
        return $this->belongsTo(Team::class);
    }

    /**
     * Relasi ke jenis surat.
     */
    public function type()
    {
        return $this->belongsTo(LetterType::class, 'letter_type_id');
    }

    /**
     * Relasi ke warga (pemohon).
     */
    public function resident()
    {
        return $this->belongsTo(Resident::class);
    }

    /**
     * Relasi ke surat keluar (final).
     */
    public function outgoingLetter()
    {
        return $this->belongsTo(OutgoingLetter::class);
    }

    public function histories()
    {
        return $this->hasMany(LetterRequestHistory::class);
    }

    /**
     * Generate nomor permohonan unik (resi).
     */
    public static function generateRequestNumber(): string
    {
        $prefix = 'REQ-' . now()->format('Ymd') . '-';
        $lastNumber = self::withTrashed()
            ->whereDate('created_at', now()->toDateString())
            ->count() + 1;

        return $prefix . str_pad($lastNumber, 5, '0', STR_PAD_LEFT);
    }

    /**
     * Boot model.
     */
    protected static function booted()
    {
        static::creating(function ($model) {
            if (empty($model->request_number)) {
                $model->request_number = self::generateRequestNumber();
            }
        });
    }

    /**
     * Helper untuk dapatkan label status.
     */
    public function getStatusLabelAttribute(): string
    {
        return self::STATUS[$this->status] ?? '-';
    }
}
